const mysql = require('mysql2/promise');
require('dotenv').config();

async function testUserStructure() {
  const connection = await mysql.createConnection({
    host: process.env.DB_HOST || 'localhost',
    user: process.env.DB_USER || 'root',
    password: process.env.DB_PASSWORD || '',
    database: process.env.DB_NAME || 'tuc'
  });

  try {
    console.log('Testing user table structure...');
    
    // Check table structure
    const [columns] = await connection.execute(`
      DESCRIBE users
    `);
    
    console.log('\n📋 Users table structure:');
    columns.forEach(col => {
      console.log(`  ${col.Field}: ${col.Type} ${col.Null === 'NO' ? 'NOT NULL' : 'NULL'} ${col.Default ? `DEFAULT ${col.Default}` : ''}`);
    });

    // Check if we have any users
    const [users] = await connection.execute(`
      SELECT id, name, email, role, assignedShopIds, assignedShopNames, isDeleted 
      FROM users 
      WHERE isDeleted = 0 
      LIMIT 5
    `);
    
    console.log(`\n👥 Found ${users.length} active users:`);
    users.forEach(user => {
      console.log(`  ID: ${user.id}, Name: ${user.name}, Email: ${user.email}, Role: ${user.role}`);
      console.log(`      Shop IDs: ${user.assignedShopIds || '[]'}`);
      console.log(`      Shop Names: ${user.assignedShopNames || '[]'}`);
    });

    console.log('\n✅ User structure test completed successfully!');
    
  } catch (error) {
    console.error('❌ Test failed:', error.message);
    throw error;
  } finally {
    await connection.end();
  }
}

// Run the test
if (require.main === module) {
  testUserStructure()
    .then(() => {
      console.log('Test script completed');
      process.exit(0);
    })
    .catch((error) => {
      console.error('Test script failed:', error);
      process.exit(1);
    });
}

module.exports = testUserStructure;