import { DataTypes, Model, Optional } from "sequelize";
import sequelize from "../middlewears/sequelize";
import Joi from "joi";

export enum OrderType {
  BULK = "Bulk Order",
  CUSTOM = "Custom Order",
}

export enum OrderStatus {
  PENDING = "Pending",
  TAILOR = "Tailor",
  STITCHING = "Stitching",
  KAJBUTTON = "Kajbutton",
  DONE = "Done",
}

export interface IOrder {
  id: number;
  schoolId: number;
  masterId: number; // Foreign key to users table (Master role)
  name: string;
  address: string;
  phoneNumber: string;
  totalPrice: string;
  type: OrderType;
  status: OrderStatus;
  totalQuantity: number;
  isDeleted: boolean;
  createdAt: Date;
  updatedAt: Date;
}

interface IOrderAttributes extends Optional<IOrder, "id"> {}

class Order extends Model<IOrder, IOrderAttributes> implements IOrder {
  public id!: number;
  public schoolId!: number;
  public masterId!: number;
  public name!: string;
  public address!: string;
  public phoneNumber!: string;
  public totalPrice!: string;
  public type!: OrderType;
  public status!: OrderStatus;
  public totalQuantity!: number;
  public isDeleted!: boolean;
  public readonly createdAt!: Date;
  public readonly updatedAt!: Date;
}

Order.init(
  {
    id: { type: DataTypes.INTEGER, primaryKey: true, autoIncrement: true },

    schoolId: {
      type: DataTypes.INTEGER,
      references: { model: "schools", key: "id" },
      allowNull: false,
    },

    masterId: {
      type: DataTypes.INTEGER,
      references: { model: "users", key: "id" },
      allowNull: false,
    },

    name: { type: DataTypes.STRING, allowNull: false },
    address: { type: DataTypes.STRING, allowNull: false },
    phoneNumber: { type: DataTypes.STRING, allowNull: false },
    totalPrice: { type: DataTypes.STRING, allowNull: false },

    type: {
      type: DataTypes.ENUM(...Object.values(OrderType)),
      allowNull: false,
    },

    status: {
      type: DataTypes.ENUM(...Object.values(OrderStatus)),
      allowNull: false,
      defaultValue: OrderStatus.PENDING,
    },

    totalQuantity: { type: DataTypes.INTEGER, allowNull: false },

    isDeleted: {
      type: DataTypes.BOOLEAN,
      defaultValue: false,
      allowNull: false,
    },

    createdAt: {
      type: DataTypes.DATE,
      allowNull: false,
      defaultValue: DataTypes.NOW,
    },

    updatedAt: {
      type: DataTypes.DATE,
      allowNull: false,
      defaultValue: DataTypes.NOW,
    },
  },
  {
    sequelize,
    tableName: "orders",
  }
);

const OrderValidationSchema = Joi.object({
  name: Joi.string().min(1).max(255).required(),
  address: Joi.string().min(1).max(255).required(),
  phoneNumber: Joi.string().min(1).max(255).required(),
  totalPrice: Joi.string().min(1).max(255).required(),
  totalQuantity: Joi.number().min(1).required(),
  type: Joi.string().valid(OrderType.BULK, OrderType.CUSTOM).required(),
  schoolId: Joi.number().min(1).required(),
  masterId: Joi.number().min(1).required(), // Required field for Master assignment
  status: Joi.string().valid(...Object.values(OrderStatus)).optional().default(OrderStatus.PENDING),
  isDeleted: Joi.boolean().default(false).optional(),
});

export { Order, OrderValidationSchema };
