import { DataTypes, Model, Optional } from "sequelize";
import Joi from "joi";
import sequelize from "../middlewears/sequelize";

export interface IEmployee {
  id: number;
  employeeName: string;
  email: string;
  address: string;
  phoneNumber: string;
  skills: string;
  paymentType: "Daily" | "Monthly";
  price: number;
  isDeleted: boolean;
  createdAt?: Date;
  updatedAt?: Date;
}

interface IEmployeeCreationAttributes
  extends Optional<
    IEmployee,
    | "id"
    | "createdAt"
    | "updatedAt"
    | "address"
    | "phoneNumber"
    | "skills"
    | "paymentType"
    | "price"
    | "isDeleted"
  > {}

class EmployeeModel
  extends Model<IEmployee, IEmployeeCreationAttributes>
  implements IEmployee
{
  public id!: number;
  public employeeName!: string;
  public email!: string;
  public address!: string;
  public phoneNumber!: string;
  public skills!: string;
  public paymentType!: "Daily" | "Monthly";
  public price!: number;
  public isDeleted!: boolean;
  public readonly createdAt!: Date;
  public readonly updatedAt!: Date;
}

EmployeeModel.init(
  {
    id: { type: DataTypes.INTEGER, autoIncrement: true, primaryKey: true },
    employeeName: { type: DataTypes.STRING, allowNull: false },
    email: { type: DataTypes.STRING, allowNull: false, unique: true },
    address: { type: DataTypes.STRING, allowNull: true },
    phoneNumber: { type: DataTypes.STRING, allowNull: true },
    skills: { type: DataTypes.STRING, allowNull: true },
    paymentType: {
      type: DataTypes.ENUM("Daily", "Monthly"),
      allowNull: false,
      defaultValue: "Daily",
    },
    price: {
      type: DataTypes.DECIMAL(10, 2),
      allowNull: false,
      defaultValue: 0.0,
    },
    isDeleted: {
      type: DataTypes.BOOLEAN,
      allowNull: false,
      defaultValue: false,
    },
    createdAt: {
      type: DataTypes.DATE,
      allowNull: false,
      defaultValue: DataTypes.NOW,
    },
    updatedAt: {
      type: DataTypes.DATE,
      allowNull: false,
      defaultValue: DataTypes.NOW,
    },
  },
  {
    sequelize,
    tableName: "employees",
  }
);

const employeeValidationSchema = Joi.object({
  employeeName: Joi.string().min(1).max(100).required(),
  email: Joi.string().email().required(),
  address: Joi.string().min(1).max(255).optional(),
  phoneNumber: Joi.string().min(7).max(15).optional(),
  skills: Joi.string().min(1).max(500).optional(),
  paymentType: Joi.string().valid("Daily", "Monthly").optional(),
  price: Joi.number().min(0).optional(),
  isDeleted: Joi.boolean().optional(),
});

export { EmployeeModel, employeeValidationSchema };
