import { AddonModel, addonValidationSchema } from "../models/addon";
import { Request, Response } from "express";

/// Create new addon
export const createAddon = async (req: Request, res: Response) => {
  const { error } = addonValidationSchema.validate(req.body);
  if (error) return res.status(400).json({ error: error.details[0].message });

  try {
    const newAddon = await AddonModel.create(req.body);
    return res
      .status(200)
      .json({ message: "Addon created successfully.", id: newAddon.id });
  } catch (ex) {
    return res.status(500).json({ error: "An unexpected error occurred." });
  }
};

/// Get all addons
export const getAllAddons = async (req: Request, res: Response) => {
  try {
    const addons = await AddonModel.findAll({
      where: { isDeleted: false },
      order: [["createdAt", "DESC"]],
    });
    return res.status(200).json(addons);
  } catch (ex) {
    return res.status(500).json({ error: "An unexpected error occurred." });
  }
};

/// Get addon by ID
export const getAddonById = async (req: Request, res: Response) => {
  try {
    const addon = await AddonModel.findOne({
      where: { id: req.params.id, isDeleted: false },
    });
    
    if (!addon) {
      return res.status(404).json({ error: "Addon not found." });
    }
    
    return res.status(200).json(addon);
  } catch (ex) {
    return res.status(500).json({ error: "An unexpected error occurred." });
  }
};

/// Update existing addon
export const updateAddon = async (req: Request, res: Response) => {
  try {
    // Find the addon by ID
    const addon = await AddonModel.findByPk(req.params.id);
    if (!addon) return res.status(404).json({ error: "Addon not found." });

    // Check if addon is deleted
    if (addon.isDeleted) {
      return res.status(404).json({ error: "Addon not found." });
    }

    // Validate the request body
    const { error } = addonValidationSchema.validate(req.body);
    if (error) return res.status(400).json({ error: error.details[0].message });

    // Update the addon details
    await addon.update(req.body);

    // Send success response
    return res
      .status(200)
      .json({ message: "Addon updated successfully.", id: addon.id });
  } catch (ex) {
    console.error(`Error updating addon: ${ex}`);
    return res.status(500).json({ error: "An unexpected error occurred." });
  }
};

/// Delete an addon (soft delete)
export const deleteAddon = async (req: Request, res: Response) => {
  try {
    // Find the addon by ID
    const addon = await AddonModel.findByPk(req.params.id);
    if (!addon) return res.status(404).json({ error: "Addon not found." });

    // Check if addon is already deleted
    if (addon.isDeleted) {
      return res.status(404).json({ error: "Addon not found." });
    }

    // Soft delete the addon
    await addon.update({ isDeleted: true });

    // Send success response
    return res.status(200).json({ message: "Addon deleted successfully." });
  } catch (ex) {
    return res.status(500).json({ error: "An unexpected error occurred." });
  }
};