"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.deleteEmployee = exports.updateEmployee = exports.getAllEmployees = exports.createEmployee = void 0;
const employee_1 = require("../models/employee");
const sequelize_1 = require("sequelize");
/// Create new employee
const createEmployee = async (req, res) => {
    const { error } = employee_1.employeeValidationSchema.validate(req.body);
    if (error)
        return res.status(400).json({ error: error.details[0].message });
    try {
        // Check if email already exists
        const existingEmployee = await employee_1.EmployeeModel.findOne({
            where: { email: req.body.email },
        });
        if (existingEmployee)
            return res.status(400).json({ error: "Email address already exists." });
        const newEmployee = await employee_1.EmployeeModel.create(req.body);
        return res
            .status(200)
            .json({ message: "Employee created successfully.", id: newEmployee.id });
    }
    catch (ex) {
        return res.status(500).json({ error: "An unexpected error occurred." });
    }
};
exports.createEmployee = createEmployee;
/// Get all employees
const getAllEmployees = async (req, res) => {
    try {
        const employees = await employee_1.EmployeeModel.findAll({
            where: { isDeleted: false },
        });
        return res.status(200).json(employees);
    }
    catch (ex) {
        return res.status(500).json({ error: "An unexpected error occurred." });
    }
};
exports.getAllEmployees = getAllEmployees;
/// Update existing employee
const updateEmployee = async (req, res) => {
    try {
        // Find the employee by ID
        const employee = await employee_1.EmployeeModel.findByPk(req.params.id);
        if (!employee)
            return res.status(404).json({ error: "Employee not found." });
        // Check if updating email and it already exists
        if (req.body.email) {
            const existingEmployee = await employee_1.EmployeeModel.findOne({
                where: { email: req.body.email, id: { [sequelize_1.Op.ne]: req.params.id } },
            });
            if (existingEmployee)
                return res.status(400).json({ error: "Email address already exists." });
        }
        // Validate the request body
        const { error } = employee_1.employeeValidationSchema.validate(req.body);
        if (error)
            return res.status(400).json({ error: error.details[0].message });
        // Update the employee details
        await employee.update(req.body);
        // Send success response
        return res
            .status(200)
            .json({ message: "Employee updated successfully.", id: employee.id });
    }
    catch (ex) {
        console.error(`Error updating employee: ${ex}`);
        return res.status(500).json({ error: "An unexpected error occurred." });
    }
};
exports.updateEmployee = updateEmployee;
/// Delete an employee
const deleteEmployee = async (req, res) => {
    try {
        // Find the employee by ID
        const employee = await employee_1.EmployeeModel.findByPk(req.params.id);
        if (!employee)
            return res.status(404).json({ error: "Employee not found." });
        // Soft delete the employee
        await employee.update({ isDeleted: true });
        // Send success response
        return res.status(200).json({ message: "Employee deleted successfully." });
    }
    catch (ex) {
        return res.status(500).json({ error: "An unexpected error occurred." });
    }
};
exports.deleteEmployee = deleteEmployee;
