"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.deleteCustomer = exports.updateCustomer = exports.getAllCustomers = exports.createCustomer = void 0;
const customer_1 = require("../models/customer");
const sequelize_1 = require("sequelize");
/// Create new customer
const createCustomer = async (req, res) => {
    const { error } = customer_1.customerValidationSchema.validate(req.body);
    if (error)
        return res.status(400).json({ error: error.details[0].message });
    try {
        // Check if email already exists
        const existingCustomer = await customer_1.CustomerModel.findOne({
            where: { emailAddress: req.body.emailAddress },
        });
        if (existingCustomer)
            return res.status(400).json({ error: "Email address already exists." });
        const newCustomer = await customer_1.CustomerModel.create(req.body);
        return res
            .status(200)
            .json({ message: "Customer created successfully.", id: newCustomer.id });
    }
    catch (ex) {
        return res.status(500).json({ error: "An unexpected error occurred." });
    }
};
exports.createCustomer = createCustomer;
/// Get all customers
const getAllCustomers = async (req, res) => {
    try {
        const customers = await customer_1.CustomerModel.findAll({
            where: { isDeleted: false },
        });
        return res.status(200).json(customers);
    }
    catch (ex) {
        return res.status(500).json({ error: "An unexpected error occurred." });
    }
};
exports.getAllCustomers = getAllCustomers;
/// Update existing customer
const updateCustomer = async (req, res) => {
    try {
        // Find the customer by ID
        const customer = await customer_1.CustomerModel.findByPk(req.params.id);
        if (!customer)
            return res.status(404).json({ error: "Customer not found." });
        // Check if updating email and it already exists
        if (req.body.emailAddress) {
            const existingCustomer = await customer_1.CustomerModel.findOne({
                where: {
                    emailAddress: req.body.emailAddress,
                    id: { [sequelize_1.Op.ne]: req.params.id },
                },
            });
            if (existingCustomer)
                return res.status(400).json({ error: "Email address already exists." });
        }
        // Validate the request body
        const { error } = customer_1.customerValidationSchema.validate(req.body);
        if (error)
            return res.status(400).json({ error: error.details[0].message });
        // Update the customer details
        await customer.update(req.body);
        // Send success response
        return res
            .status(200)
            .json({ message: "Customer updated successfully.", id: customer.id });
    }
    catch (ex) {
        console.error(`Error updating customer: ${ex}`);
        return res.status(500).json({ error: "An unexpected error occurred." });
    }
};
exports.updateCustomer = updateCustomer;
/// Delete a customer
const deleteCustomer = async (req, res) => {
    try {
        // Find the customer by ID
        const customer = await customer_1.CustomerModel.findByPk(req.params.id);
        if (!customer)
            return res.status(404).json({ error: "Customer not found." });
        // Soft delete the customer
        await customer.update({ isDeleted: true });
        // Send success response
        return res.status(200).json({ message: "Customer deleted successfully." });
    }
    catch (ex) {
        return res.status(500).json({ error: "An unexpected error occurred." });
    }
};
exports.deleteCustomer = deleteCustomer;
